"use client";

import React, { useRef } from 'react';
import { useRouter } from 'next/navigation';
import NavbarCommon from "@/component/shared/NavbarCommon";
import { PageFooter } from "@/app/twofactauth/page";

const Page = () => {
    const router = useRouter();
    const emailRef = useRef<HTMLInputElement>(null);

    const handleSubmit = (e: React.FormEvent<HTMLFormElement>) => {
        e.preventDefault();

        const form = e.currentTarget;
        if (!form.checkValidity()) {
            form.reportValidity();
            return;
        }

        const email = (emailRef.current?.value || "").trim();
        if (!email) return;

        // Optional: keep for later steps
        if (typeof window !== "undefined") {
            localStorage.setItem("twofact_email", email);
        }

        // Navigate to authmail/[email]/page.tsx (relative to current segment)
        router.push(`authmail/${encodeURIComponent(email)}`);
    };

    return (
        <>
            <NavbarCommon />

            {/* dark strip heading right after navbar */}
            <div className="w-full bg-[#2d4b6a] text-white">
                <div className="max-w-7xl mx-auto px-6 md:px-8 py-3">
                    <h2 className="text-2xl md:text-3xl font-bold">
                        Two-factor authentication
                    </h2>
                </div>
            </div>

            {/* sub-nav bar — screenshot style */}
            <div className="w-full bg-[#e9edf1] border-t-[3px] border-[#2d4b6a] border-b">
                <div className="max-w-7xl mx-auto px-6 md:px-8">
                    <nav aria-label="Secondary">
                        <ul className="flex text-sm font-semibold text-[#2d4b6a] divide-x divide-gray-300">
                            <li className="border-l border-gray-300">
                                <a href="#" className="block px-6 py-3 hover:underline focus:underline focus:outline-none">
                                    Contact us
                                </a>
                            </li>
                            <li>
                                <a href="#" className="block px-6 py-3 hover:underline focus:underline focus:outline-none">
                                    FAQ
                                </a>
                            </li>
                            <li>
                                <a href="#" className="block px-6 py-3 hover:underline focus:underline focus:outline-none">
                                    Help
                                </a>
                            </li>
                            <li className="flex-1" />
                        </ul>
                    </nav>
                </div>
            </div>

            {/* main content */}
            <main className="max-w-7xl mx-auto px-6 md:px-8 py-8 md:py-10">
                <section aria-labelledby="register-email-heading">
                    {/* Page title + thin red rule (matches screenshot) */}
                    <h1 id="register-email-heading" className="text-2xl md:text-3xl font-bold text-gray-900">
                        Register your email address
                    </h1>
                    <div className="mt-2 h-[2px] w-full bg-[#a71e32]" />

                    {/* lead-in copy */}
                    <p id="register-email-lead" className="mt-6 text-gray-800">
                        Use your email address as a second factor.
                    </p>
                    <p className="mt-4 text-gray-800">
                        Enter your email address. A code will be sent to that email to
                        complete the two-factor authentication. You will need to verify
                        your email address before registration is complete.
                    </p>

                    {/* form */}
                    <form
                        onSubmit={handleSubmit}
                        className="mt-8 space-y-6"
                        aria-describedby="register-email-lead"
                        noValidate={false}
                    >
                        <div>
                            <label htmlFor="email" className="block text-base font-semibold text-gray-900">
                                <span className="text-red-600 mr-1" aria-hidden="true">*</span>
                                Email address
                                <span className="font-normal text-gray-600"> (Required field)</span>
                            </label>

                            <input
                                id="email"
                                name="email"
                                type="email"
                                required
                                autoComplete="email"
                                ref={emailRef}
                                className="mt-2 block w-full max-w-xs rounded border border-gray-300 px-3 py-2 text-gray-900 shadow-sm placeholder-gray-400 focus:outline-none focus:ring-2 focus:ring-[#2d4b6a] focus:border-[#2d4b6a]"
                                aria-required="true"
                            />
                        </div>

                        {/* primary action */}
                        <div>
                            <button
                                type="submit"
                                className="inline-flex items-center rounded bg-[#2d4b6a] px-4 py-2 text-sm font-semibold text-white shadow-sm hover:bg-[#244059] focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-[#2d4b6a]"
                            >
                                Continue
                            </button>
                        </div>

                        {/* secondary actions (separate row, per screenshot) */}
                        <div className="flex items-center gap-3">
                            <a
                                href="#"
                                className="inline-flex items-center rounded border border-gray-300 bg-gray-100 px-4 py-2 text-sm font-medium text-gray-800 hover:bg-gray-200 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-[#2d4b6a]"
                                role="button"
                            >
                                Back
                            </a>
                            <a
                                href="#"
                                className="inline-flex items-center rounded border border-gray-300 bg-gray-100 px-4 py-2 text-sm font-medium text-gray-800 hover:bg-gray-200 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-[#2d4b6a]"
                                role="button"
                            >
                                Cancel
                            </a>
                        </div>
                    </form>

                    {/* footer meta line */}
                    <p className="mt-10 text-sm text-gray-600">Date modified: 2024-06-15</p>
                </section>
            </main>

            <PageFooter />
        </>
    );
};

export default Page;
