"use client";
import React, { JSX, useState } from "react";
import { FaCaretRight, FaCaretDown } from "react-icons/fa";

const pageData = {
    mainHeading: "Check if this is the right account for you",
    accordionData: [
        {
            id: "apply",
            title: "Apply for these applications",
            content: {
                list1: [
                    "work permits",
                    "Express Entry (immigrating as a skilled worker)",
                    "International Experience Canada",
                    "extending work and study permits",
                    "visitor records (to stay in Canada longer)",
                    "change your school (if you have a study permit)",
                    "change conditions on your work permit",
                    "Canadian citizenship certificates (proof of citizenship)",
                    "search of citizenship records",
                    "temporary health benefits for refugees and protected persons (the Interim Federal Health Program)",
                ],
                paragraph1:
                    "You might also need to use this account for these applications, depending on your situation:",
                list2: [
                    {
                        text: "study permits, including those under the Francophone Minority Communities Student Pilot (FMCSP) (",
                        bold: "only if",
                        rest: " you’re applying with a family member who is applying for a work permit)",
                    },
                    {
                        text: "some visitor visas (",
                        link: {
                            text: "get visitor visa instructions for your situation",
                            href: "#",
                        },
                        rest: ")",
                    },
                ],
                paragraph2: `If your application isn’t listed, review your program content to get instructions on how to apply:`,
                list3: [
                    {
                        link: {
                            text: "Visit Canada",
                            href: "#",
                        },
                        rest: " (includes visas and electronic travel authorizations)",
                    },
                    {
                        link: {
                            text: "Immigrate to Canada",
                            href: "#",
                        },
                        rest: " (includes family sponsorship, Express entry and other economic immigration programs)",
                    },
                    {
                        link: {
                            text: "Work in Canada",
                            href: "#",
                        },
                        rest: " (includes work permits and International Experience Canada)",
                    },
                    {
                        link: {
                            text: "Study in Canada",
                            href: "#",
                        },
                        rest: " (includes study permits and extensions)",
                    },
                    {
                        link: {
                            text: "Canadian citizenship",
                            href: "#",
                        },
                        rest: " (includes applying for citizenship and proof of citizenship)",
                    },
                    {
                        link: {
                            text: "Refugees and asylum",
                            href: "#",
                        },
                        rest: " (includes asylum claims and sponsoring refugees)",
                    },
                ],
            },
        },
        {
            id: "status",
            title: "Check the status of these applications",
            content: {
                list1: [
                    "applications you submitted in this account",
                    {
                        text: "these applications (after you ",
                        link: {
                            text: "link the application to your account",
                            href: "#",
                        },
                        rest: "):",
                        sublist: [
                            "Agri-Food Pilot",
                            "Atlantic Immigration Pilot",
                            "Atlantic Immigration Program",
                            "Caring for children",
                            "Caring for people with high medical needs",
                            "electronic travel authorization (eTA)",
                            "Express Entry",
                            "family sponsorship (excluding applications to become a sponsor)",
                            "Home Child Care Provider Pilot",
                            "Home Support Worker Pilot",
                            "in-Canada live-in caregivers",
                            "in-Canada protected persons and convention refugees (excluding dependant applications)",
                            "Interim Federal Health Program",
                            "International Experience Canada (IEC)",
                            "permanent resident card",
                            "Provincial Nominee Program",
                            "Quebec entrepreneurs",
                            "Quebec investors",
                            "Quebec-selected skilled worker",
                            "Rural and Northern Immigration Pilot",
                            "self-employed persons (including Quebec)",
                            "start-up visa",
                            "study permit and extension",
                            "Temporary Resident to Permanent Resident Pathway (TR to PR Pathway)",
                            "visitor visa",
                            "visitor record",
                            "work permit and extension",
                        ],
                    },
                ],
                paragraph1: {
                    text: "Can’t find your application? ",
                    link: {
                        text: "Get instructions on how to check your application status.",
                        href: "#",
                    },
                },
            },
        },
        {
            id: "upload",
            title: "Upload requested documents for these applications",
            content: {
                list1: [
                    {
                        text: "applications you submitted in this account (",
                        bold: "only if",
                        rest: " we ask you for the document)",
                    },
                    {
                        text: "electronic travel authorizations (eTA) (",
                        bold: "only if",
                        rest: " we email you about submitting your documents)",
                    },
                ],
                paragraph1: {
                    text: "",
                    link: {
                        text: "Get more help with submitting documents we request.",
                        href: "#",
                    },
                },
            },
        },
    ],
};

const Link: React.FC<{ href?: string; children: React.ReactNode }> = ({
                                                                          href = "#",
                                                                          children,
                                                                      }) => (
    <a
        href={href}
        className="text-[#26374a] underline hover:text-blue-700 transition-colors"
    >
        {children}
    </a>
);

const ComplexListItem: React.FC<{ item: any }> = ({ item }) => {
    if (typeof item === "string") return <li className="mb-1">{item}</li>;

    return (
        <li className="mb-2 leading-relaxed">
            {item.text}
            {item.bold && <span className="font-semibold">{item.bold}</span>}
            {item.link && <Link href={item.link.href}>{item.link.text}</Link>}
            {item.rest}
            {item.sublist && (
                <ul className="list-disc list-inside ml-6 mt-2 space-y-1">
                    {item.sublist.map((subItem: string, index: number) => (
                        <li key={index}>{subItem}</li>
                    ))}
                </ul>
            )}
        </li>
    );
};

export const AccordionItem: React.FC<{
    section: any;
    isOpen: boolean;
    onToggle: (id: string) => void;
    contentRenderer: (content: any) => JSX.Element;
}> = ({ section, isOpen, onToggle, contentRenderer }) => (
    <div
        className={`border ${
            isOpen ? "border-[#26374a]" : "border-gray-300"
        } rounded-none`}
    >
        <button
            onClick={() => onToggle(section.id)}
            className={`w-full px-2 py-1 text-left flex items-center space-x-3 bg-gray-50 hover:bg-gray-100 transition-colors group ${
                isOpen ? "text-blue-700" : "text-[#26374a]"
            }`}
            aria-expanded={isOpen}
        >
            {isOpen ? (
                <FaCaretDown className="text-xl transition-colors group-hover:text-blue-700" />
            ) : (
                <FaCaretRight className="text-xl transition-colors group-hover:text-blue-700" />
            )}
            <span
                className={`hover:underline font-semibold text-lg ${
                    isOpen ? "text-blue-700" : "text-[#26374a]"
                } group-hover:text-blue-700`}
            >
        {section.title}
      </span>
        </button>
        {isOpen && contentRenderer(section.content)}
    </div>
);

const ApplicationInfo: React.FC = () => {
    const [openSections, setOpenSections] = useState<Record<string, boolean>>({});

    const toggleSection = (sectionId: string) => {
        setOpenSections((prev) => ({
            ...prev,
            [sectionId]: !prev[sectionId],
        }));
    };

    const renderContent = (content: any) => (
        <div className="p-6 border-t border-gray-300 bg-white space-y-4 text-[#26374a]">
            {content.list1 && (
                <ul className="list-disc list-inside space-y-2 pl-2">
                    {content.list1.map((item: any, index: number) => (
                        <ComplexListItem key={index} item={item} />
                    ))}
                </ul>
            )}
            {content.paragraph1 && (
                <p className="mt-4 leading-relaxed">
                    {typeof content.paragraph1 === "string"
                        ? content.paragraph1
                        : content.paragraph1.text}
                    {content.paragraph1.link && (
                        <Link href={content.paragraph1.link.href}>
                            {content.paragraph1.link.text}
                        </Link>
                    )}
                </p>
            )}
            {content.list2 && (
                <ul className="list-disc list-inside space-y-2 mt-3 pl-2">
                    {content.list2.map((item: any, index: number) => (
                        <ComplexListItem key={index} item={item} />
                    ))}
                </ul>
            )}
            {content.paragraph2 && (
                <p className="mt-4 leading-relaxed">{content.paragraph2}</p>
            )}
            {content.list3 && (
                <ul className="list-disc list-inside space-y-2 mt-3 pl-2">
                    {content.list3.map((item: any, index: number) => (
                        <li key={index} className="leading-relaxed">
                            {item.link && <Link href={item.link.href}>{item.link.text}</Link>}
                            {item.rest}
                        </li>
                    ))}
                </ul>
            )}
        </div>
    );

    return (
        <div className="max-w-7xl mx-auto py-6 font-sans text-gray-800 space-y-6  px-4 sm:px-6 lg:px-8">
            <h2 className="text-3xl font-semibold mb-6 text-[#26374a]">
                {pageData.mainHeading}
            </h2>

            <div className="space-y-5">
                {pageData.accordionData.map((section) => (
                    <AccordionItem
                        key={section.id}
                        section={section}
                        isOpen={!!openSections[section.id]}
                        onToggle={toggleSection}
                        contentRenderer={renderContent}
                    />
                ))}
            </div>
        </div>
    );
};

export default ApplicationInfo;
