"use client";
import React, { JSX, ReactNode, useState } from "react";
import { FaCaretRight, FaCaretDown } from "react-icons/fa";

const ToggleDetails: React.FC<{
    summary: ReactNode;
    children?: ReactNode;
    startOpen?: boolean;
    className?: string;
}> = ({ summary, children, startOpen = false, className = "" }) => {
    const [open, setOpen] = useState<boolean>(startOpen);

    return (
        <div className={`mb-3 ${className}`}>
            <button
                type="button"
                onClick={() => setOpen((v) => !v)}
                className={[
                    "w-full flex items-center gap-2",
                    "px-2 py-2 text-left",
                    " border border-gray-300",
                    "hover:bg-gray-50 hover:border-blue-700 transition-colors",
                    open ? "text-blue-700" : "text-[#26374a]",
                    "focus:outline-none focus-visible:outline focus-visible:outline-2",
                    "focus-visible:outline-blue-700 focus-visible:outline-offset-2",
                    "group",
                ].join(" ")}
                aria-expanded={open}
            >
                {open ? (
                    <FaCaretDown className="text-xl transition-colors group-hover:text-blue-700" />
                ) : (
                    <FaCaretRight className="text-xl transition-colors group-hover:text-blue-700" />
                )}
                <span
                    className={[
                        "hover:underline font-normal text-[16px] leading-snug",
                        open ? "text-blue-700" : "text-[#26374a]",
                        "group-hover:text-blue-700",
                    ].join(" ")}
                >
          {summary}
        </span>
            </button>

            {open && (
                <div className="border border-t-0 border-gray-300 rounded-b   p-4 text-[#26374a] leading-relaxed">
                    {children}
                </div>
            )}
        </div>
    );
};

/* Top-level component */
export default function HelpWithYourAccount(): JSX.Element {
    return (
        <section className="w-full bg-gray-100">
            <div className="max-w-7xl mx-auto px-4 md:px-6 py-8 md:py-10">
                <h2 id="help" className="text-3xl font-bold mb-4 md:mb-6 text-[#26374a]">
                    Help with your account
                </h2>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-3 md:gap-4 wb-eqht-grd">
                    {/* LEFT COLUMN */}
                    <div>
                        {/* Errors and issues when you sign in */}
                        <ToggleDetails summary={<span id="errors">Errors and issues when you sign in</span>}>
                            <p>
                                You can use the same GCKey username and password on different computers, phones and
                                tablets.
                            </p>

                            <ToggleDetails summary={"If you're getting error messages:"}>
                                <ul className="list-disc ml-6 space-y-2">
                                    <li>
                                        Use the{" "}
                                        <a
                                            className="text-[#26374a] hover:text-blue-700 underline"
                                            href="https://ircc.canada.ca/english/helpcentre/answer.asp?qnum=316&top=23"
                                        >
                                            most recent version of a supported web browser
                                        </a>{" "}
                                        like
                                        <ul className="list-disc ml-6 mt-2">
                                            <li>Safari</li>
                                            <li>Chrome</li>
                                            <li>Firefox</li>
                                            <li>Microsoft Edge</li>
                                        </ul>
                                        <p className="mt-2">
                                            <strong>If you're using Internet Explorer</strong>, update your browser to
                                            Microsoft Edge.
                                        </p>
                                    </li>

                                    <li>
                                        Don't use a favourite link or bookmark to pages inside the account. You must
                                        sign in to see them.
                                    </li>

                                    <li>Use your browser's default settings. GCKey needs JavaScript and cookies enabled.</li>

                                    <li>
                                        Update your browser if you see one of these messages when you try to sign in:
                                        <ul className="list-disc ml-6 mt-2">
                                            <li>Page Can’t Be Displayed</li>
                                            <li>Cannot Establish a Secure Connection</li>
                                            <li>Page Not Available</li>
                                        </ul>
                                        <p className="mt-2">
                                            You can usually find the option to update your browser in your browser menu's
                                            settings or preferences.
                                        </p>
                                    </li>
                                </ul>
                            </ToggleDetails>

                            <ToggleDetails summary={"Other technical issues"}>
                                <ul className="list-disc ml-6 space-y-4">
                                    <li>
                                        <strong>Too many attempts</strong>
                                        <p className="mt-1">
                                            You may get a failure message if you make too many unsuccessful attempts to
                                            sign in during the same session. Close your browser window or tab and try
                                            again.
                                        </p>
                                    </li>

                                    <li>
                                        <strong>Experiencing technical difficulties</strong>
                                        <p className="mt-1">
                                            The site could be down for a number of reasons, including for system
                                            maintenance. We apologize for the inconvenience. Try again later.
                                        </p>
                                    </li>

                                    <li>
                                        <strong>Session expired</strong>
                                        <p className="mt-1">
                                            For security, your session will be closed if it's inactive for a certain number
                                            of minutes. Sign in again.
                                        </p>
                                    </li>
                                </ul>
                            </ToggleDetails>

                            <p className="mt-3">
                                <a
                                    className="text-[#26374a] hover:text-blue-700 underline"
                                    href="https://ircc.canada.ca/english/helpcentre/answer.asp?qnum=1491&top=23.3"
                                >
                                    Get more help with sign-in issues
                                </a>
                            </p>
                        </ToggleDetails>

                        {/* GCKey two-factor authentication */}
                        <ToggleDetails summary={<span id="mfa">GCKey two-factor authentication</span>}>
                            <p>
                                Two-factor authentication protects your account. You need to choose a second
                                authentication method (other than your username and password) to use each time you
                                sign in.
                            </p>
                            <p className="mt-2">
                                <a
                                    className="text-[#26374a] hover:text-blue-700 underline"
                                    href="/en/immigration-refugees-citizenship/services/application/multi-factor-authentication.html#register"
                                >
                                    Set up your two-factor authentication method
                                </a>
                            </p>

                            <h3 className="font-semibold mt-4">Change your two-factor authentication method</h3>
                            <p className="mt-2">
                                You need access to your existing email or authenticator app, or to your saved recovery
                                codes.
                            </p>

                            <ToggleDetails summary={"If you have access to your two-factor authentication method"}>
                                <ol className="list-decimal ml-6 space-y-2">
                                    <li>Sign in to your account with your GCKey username and password.</li>
                                    <li>
                                        Select <strong>Continue</strong> on the <strong>Welcome</strong> page.
                                    </li>
                                    <li>Use your current two-factor method.</li>
                                    <li>
                                        On the <strong>Authentication success</strong> page, select{" "}
                                        <strong>Change your two-factor authentication</strong>.
                                    </li>
                                    <li>
                                        On the <strong>Two-factor authentication</strong> page, follow the prompts to set
                                        up two-factor authentication again.
                                    </li>
                                    <li>
                                        Save your new recovery codes somewhere safe. Your old codes will no longer work.
                                    </li>
                                </ol>
                            </ToggleDetails>

                            <ToggleDetails summary={"If you don't have access to your two-factor authentication method"}>
                                <details>
                                    <summary className="sr-only">helper</summary>
                                </details>

                                <ToggleDetails summary={"You have recovery codes:"}>
                                    <ol className="list-decimal ml-6 space-y-2">
                                        <li>Sign in to your account with your GCKey username and password.</li>
                                        <li>Select <strong>Recover account</strong> on the two-factor authentication page.</li>
                                        <li>
                                            Enter one of the recovery codes you saved when you first set up two-factor
                                            authentication.
                                        </li>
                                        <li>
                                            Select <strong>Change your two-factor authentication</strong> on the
                                            authentication success page.
                                        </li>
                                        <li>Follow the prompts to set up two-factor authentication again.</li>
                                        <li>
                                            Save your new recovery codes somewhere safe. Your old codes will no longer work.
                                        </li>
                                    </ol>
                                </ToggleDetails>

                                <ToggleDetails summary={"You don’t have recovery codes:"}>
                                    <ol className="list-decimal ml-6 space-y-2">
                                        <li>Sign up for a new GCKey username and password.</li>
                                        <li>Create a new account in the IRCC secure account.</li>
                                        <li>Follow the prompts to set up two-factor authentication again.</li>
                                        <li>
                                            Save your new recovery codes somewhere safe. Your old codes will no longer work.
                                        </li>
                                    </ol>
                                    <p className="mt-2">During this process, you may be able to:</p>
                                    <ul className="list-disc ml-6">
                                        <li>recover your account</li>
                                        <li>
                                            <a
                                                className="text-[#26374a] hover:text-blue-700 underline"
                                                href="/en/immigration-refugees-citizenship/services/application/account/link-paper-online.html"
                                            >
                                                link your existing applications to your new account
                                            </a>
                                        </li>
                                    </ul>
                                </ToggleDetails>
                            </ToggleDetails>

                            <ToggleDetails summary={"Contact the GCKey Service Desk"}>
                                <p>
                                    <strong>
                                        GCKey representatives can only help with GCKey issues (including two-factor
                                        authentication)
                                    </strong>
                                    . They don’t have access to any Government of Canada account or application
                                    information.
                                </p>
                                <ul className="list-disc ml-6 mt-2">
                                    <li>Canada and the United States: 1-855-438-1102</li>
                                    <li>Text telephone (TTY/TDD): 1-855-438-1103</li>
                                    <li>Outside Canada and the United States: 1-800-2318-6290</li>
                                </ul>
                                <p className="mt-2">
                                    GCKey representatives are available to assist you by phone year round, 24 hours a
                                    day, 7 days a week.
                                </p>
                            </ToggleDetails>

                            <p className="mt-3">
                                <a
                                    className="text-[#26374a] hover:text-blue-700 underline"
                                    href="/en/immigration-refugees-citizenship/services/application/multi-factor-authentication.html"
                                >
                                    Get more help with two-factor authentication
                                </a>
                            </p>
                        </ToggleDetails>

                        {/* Change your Sign-In Partner */}
                        <ToggleDetails summary={<span id="switch">Change your Sign-In Partner</span>}>
                            <p>
                                If you moved to a new bank or credit union, you may be able to switch your Sign-In
                                Partner:
                            </p>
                            <ol className="list-decimal ml-6 mt-2">
                                <li>
                                    Select <strong>Canadian <i>Interac</i>® Sign-In Partner</strong>.
                                </li>
                                <li>
                                    Select <strong>Switch My Sign-In Partner</strong> from the menu on the{" "}
                                    <strong>Select a Sign-In Partner</strong> page.
                                </li>
                                <li>Follow the steps to change your Sign-In Partner.</li>
                            </ol>
                        </ToggleDetails>

                        {/* If your personal reference code doesn’t work */}
                        <ToggleDetails summary={"If your personal reference code doesn’t work"}>
                            <p>
                                You may have a personal reference code (from the{" "}
                                <a
                                    className="text-[#26374a] hover:text-blue-700 underline"
                                    href="/en/immigration-refugees-citizenship/services/come-canada-tool.html"
                                >
                                    Come to Canada tool
                                </a>
                                ) for these applications:
                            </p>
                            <ul className="list-disc ml-6">
                                <li>Express Entry (to immigrate as a skilled worker)</li>
                                <li>International Experience Canada (IEC)</li>
                                <li>study permits (and extensions)</li>
                                <li>work permits (and extensions)</li>
                                <li>visitor visas and visitor records</li>
                            </ul>

                            <p className="mt-2">
                                Make sure you enter the code correctly in your account. Your personal reference code
                                will expire 60 days after you get it.
                            </p>
                            <p className="mt-2">
                                You <strong>don’t</strong> need a personal reference code to start an application in
                                the IRCC secure account.
                            </p>

                            <h3 className="font-semibold mt-4">Express Entry and International Experience Canada</h3>
                            <p className="mt-2">
                                The code <strong>won’t</strong> work if you created a profile in the account{" "}
                                <strong>before</strong> using the Come to Canada tool. You can still continue your
                                profile without the code, but you will need to enter your information again.
                            </p>
                        </ToggleDetails>
                    </div>

                    {/* RIGHT COLUMN */}
                    <div>
                        {/* You forgot your GCKey password or username */}
                        <ToggleDetails summary={<span id="forgot">You forgot your GCKey password or username</span>}>
                            <p>Read the instructions for your situation.</p>

                            <ToggleDetails summary={"Forgot your username and password"}>
                                <ol className="list-decimal ml-6 space-y-2">
                                    <li>
                                        Follow the{" "}
                                        <a className="text-[#26374a] hover:text-blue-700 underline" href="#username">
                                            steps to recover your username first
                                        </a>
                                        .
                                    </li>
                                    <li>
                                        Once you have your username, follow the{" "}
                                        <a className="text-[#26374a] hover:text-blue-700 underline" href="#password">
                                            steps to recover your password
                                        </a>
                                        .
                                    </li>
                                </ol>

                                <ToggleDetails summary={"You lost or no longer use the email you registered with"}>
                                    <ol className="list-decimal ml-6 space-y-2">
                                        <li>Sign up for a new GCKey username and password.</li>
                                        <li>Create a new account in the IRCC secure account.</li>
                                    </ol>
                                    <p className="mt-2">
                                        You may not be able to find your application in your new account. You may need to{" "}
                                        <a
                                            className="text-[#26374a] hover:text-blue-700 underline"
                                            href="/en/immigration-refugees-citizenship/services/application/account/link-paper-online.html"
                                        >
                                            link your application to your new account
                                        </a>
                                        .
                                    </p>
                                </ToggleDetails>
                            </ToggleDetails>

                            <ToggleDetails summary={<span id="username">Forgot your username</span>}>
                                <ToggleDetails summary={"If you have a recovery email address"}>
                                    <ol className="list-decimal ml-6 space-y-2">
                                        <li>
                                            Select <strong>GCKey username and password</strong>.
                                        </li>
                                        <li>
                                            Select <strong>Forgot your username?</strong> on the{" "}
                                            <strong>Welcome to GCKey</strong> page.
                                        </li>
                                        <li>
                                            Enter your recovery email address on the{" "}
                                            <strong>Recover Username Step 1 of 2</strong> page.
                                            <ul className="list-disc ml-6 mt-2">
                                                <li>
                                                    You provided your recovery email address either when you signed up for a
                                                    GCKey username and password or at a later time.
                                                </li>
                                            </ul>
                                        </li>
                                        <li>Select <strong>Continue</strong>.</li>
                                        <li>Select <strong>Continue</strong> again.</li>
                                        <li>
                                            Check your email. You should get an email from the Government of Canada with
                                            your GCKey username.
                                        </li>
                                        <li>
                                            Sign in with your username and password on the{" "}
                                            <strong>Welcome to GCKey</strong> page.
                                        </li>
                                    </ol>
                                </ToggleDetails>

                                <ToggleDetails summary={"If you don’t have a recovery email address"}>
                                    <ol className="list-decimal ml-6 space-y-2">
                                        <li>
                                            Select <strong>GCKey username and password</strong>.
                                        </li>
                                        <li>
                                            Select <strong>Sign up</strong> on the <strong>Welcome to GCKey</strong> page to
                                            create a new GCKey username and password.
                                        </li>
                                        <li>Create a new account in the IRCC secure account.</li>
                                    </ol>
                                    <p className="mt-2">During this process, you may be able to:</p>
                                    <ul className="list-disc ml-6">
                                        <li>recover your account</li>
                                        <li>
                                            <a
                                                className="text-[#26374a] hover:text-blue-700 underline"
                                                href="/en/immigration-refugees-citizenship/services/application/account/link-paper-online.html"
                                            >
                                                link your existing applications to your new account
                                            </a>
                                        </li>
                                    </ul>
                                </ToggleDetails>
                            </ToggleDetails>

                            <ToggleDetails summary={<span id="password">Forgot your password</span>}>
                                <ol className="list-decimal ml-6 space-y-2">
                                    <li>
                                        Select <strong>GCKey username and password</strong>.
                                    </li>
                                    <li>
                                        Select <strong>Forgot your password?</strong> on the{" "}
                                        <strong>Welcome to GCKey</strong> page.
                                    </li>
                                    <li>Enter your GCKey username.</li>
                                    <li>
                                        Select <strong>Continue</strong>.
                                        <ul className="list-disc ml-6 mt-2">
                                            <li>
                                                If you used an email address when signing up for your GCKey username,{" "}
                                                <a className="text-[#26374a] hover:text-blue-700 underline" href="#pw-reset-email">
                                                    reset your password with your email address
                                                </a>
                                                .
                                            </li>
                                            <li>
                                                If you didn't use an email address to sign up,{" "}
                                                <a
                                                    className="text-[#26374a] hover:text-blue-700 underline"
                                                    href="#pw-reset-recovery"
                                                >
                                                    reset your password using recovery questions
                                                </a>
                                                .
                                            </li>
                                        </ul>
                                    </li>
                                </ol>

                                <ToggleDetails summary={<span id="pw-reset-email">Password reset using an email address</span>}>
                                    <ol className="list-decimal ml-6 space-y-2">
                                        <li>Enter your email address.</li>
                                        <li>
                                            Check your email account. You should get an email from the Government of Canada
                                            with a confirmation code.
                                        </li>
                                        <li>
                                            Enter the confirmation code on the <strong>Forgot Your Password Step 2 of 4</strong> page.
                                        </li>
                                        <li>Select <strong>Continue</strong>.</li>
                                        <li>
                                            Enter the answer to the recovery question you’re prompted with in the{" "}
                                            <strong>My Recovery Answer</strong> field.
                                        </li>
                                        <li>Select <strong>Continue</strong>.</li>
                                        <li>
                                            Enter a new password in the <strong>New Password</strong> field.
                                        </li>
                                        <li>
                                            Re-enter that same password in the <strong>Confirm New Password</strong> field.
                                        </li>
                                        <li>Select <strong>Continue</strong>.</li>
                                        <li>
                                            Select <strong>Continue</strong> again on the{" "}
                                            <strong>Forget Your Password Complete</strong> page.
                                        </li>
                                    </ol>
                                </ToggleDetails>

                                <ToggleDetails summary={<span id="pw-reset-recovery">Password reset using recovery questions</span>}>
                                    <ol className="list-decimal ml-6 space-y-2">
                                        <li>
                                            Select <strong>Recovery Q&amp;A</strong> in the{" "}
                                            <strong>Recovery Questions and Answers</strong> box on the{" "}
                                            <strong>Forget Your Password Step 2 of 4</strong> page.
                                        </li>
                                        <li>
                                            Enter your answers in the <strong>My Recovery Answer</strong>,{" "}
                                            <strong>My Memorable Person Answer</strong> and <strong>My Memorable Date</strong> fields.
                                            <ul className="list-disc ml-6 mt-2">
                                                <li>You should find the questions and hints above each field.</li>
                                            </ul>
                                        </li>
                                        <li>
                                            Enter a new password in the <strong>Create Your New Password</strong> field.
                                        </li>
                                        <li>
                                            Enter the same password in the <strong>Confirm Your New Password</strong> field.
                                        </li>
                                        <li>Select <strong>Continue</strong>.</li>
                                        <li>
                                            Select <strong>Continue</strong> again on the{" "}
                                            <strong>Forget Your Password Complete</strong> page.
                                        </li>
                                    </ol>
                                </ToggleDetails>
                            </ToggleDetails>

                            <p className="mt-3">
                                <a
                                    className="text-[#26374a] hover:text-blue-700 underline"
                                    href="/en/government/sign-in-online-account/gckey.html"
                                >
                                    Get more help with GCKey
                                </a>
                            </p>
                        </ToggleDetails>

                        {/* GCKey revoked */}
                        <ToggleDetails summary={<span id="revoked">GCKey revoked</span>}>
                            <p>
                                You can’t recover your GCKey username and password if they were revoked. They can be
                                revoked if:
                            </p>
                            <ul className="list-disc ml-6 mt-2">
                                <li>your username and password are compromised</li>
                                <li>
                                    you haven’t used GCKey in the last 2 years
                                    <ul className="list-disc ml-6 mt-1">
                                        <li>
                                            Don’t use the same username and password you use for GCKey for any other online
                                            accounts.
                                        </li>
                                    </ul>
                                </li>
                            </ul>
                            <p className="mt-2">
                                If your GCKey username and password were revoked, you can choose a different Sign-In
                                method or sign up for a GCKey username and password again.
                            </p>
                            <p className="mt-2">To sign up for a new GCKey username and password:</p>
                            <ol className="list-decimal ml-6">
                                <li>
                                    Select <strong>GCKey username and password</strong>.
                                </li>
                                <li>
                                    Select <strong>Sign up</strong> on the <strong>Welcome to GCKey</strong> page to
                                    create a new GCKey username and password.
                                </li>
                                <li>Create a new account in the IRCC secure account.</li>
                            </ol>
                        </ToggleDetails>

                        <ToggleDetails summary={"If you don’t find your application in your account"}>
                            <p>This can happen if you:</p>
                            <ul className="list-disc ml-6">
                                <li>created a new GCKey username and password</li>
                                <li>switched to a different Sign-In Partner</li>
                            </ul>
                            <p className="mt-2">
                                To fix this, you need to{" "}
                                <a
                                    className="text-[#26374a] hover:text-blue-700 underline"
                                    href="/en/immigration-refugees-citizenship/services/application/account/link-paper-online.html"
                                >
                                    link the application to your account
                                </a>
                                .
                            </p>
                        </ToggleDetails>

                        <ToggleDetails summary={"More help options"}>
                            <ToggleDetails summary={"Help Centre"}>
                                <p>Find answers to questions about</p>
                                <ul className="list-disc ml-6">
                                    <li>
                                        <a
                                            className="text-[#26374a] hover:text-blue-700 underline"
                                            href="https://ircc.canada.ca/english/helpcentre/results-by-topic.asp?st=23.3"
                                        >
                                            common technical issues
                                        </a>
                                    </li>
                                    <li>
                                        <a
                                            className="text-[#26374a] hover:text-blue-700 underline"
                                            href="https://ircc.canada.ca/english/helpcentre/results-by-topic.asp?st=23.2"
                                        >
                                            GCKey and Sign-In Partner
                                        </a>
                                    </li>
                                    <li>
                                        <a
                                            className="text-[#26374a] hover:text-blue-700 underline"
                                            href="https://ircc.canada.ca/english/helpcentre/index-featured-can.asp"
                                        >
                                            other topics
                                        </a>
                                    </li>
                                </ul>
                            </ToggleDetails>

                            <ToggleDetails summary={"GCKey"}>
                                <p>For more information and help:</p>
                                <ul className="list-disc ml-6">
                                    <li>
                                        read
                                        <ul className="list-disc ml-6">
                                            <li>
                                                <a
                                                    className="text-[#26374a] hover:text-blue-700 underline"
                                                    href="/en/government/sign-in-online-account/gckey.html"
                                                >
                                                    the GCKey help page
                                                </a>
                                            </li>
                                            <li>
                                                <a
                                                    className="text-[#26374a] hover:text-blue-700 underline"
                                                    href="https://ircc.canada.ca/english/helpcentre/results-by-topic.asp?st=23.2.2"
                                                >
                                                    answers to common GCKey questions
                                                </a>
                                            </li>
                                        </ul>
                                    </li>
                                    <li>
                                        review the help options in the menu after you select{" "}
                                        <strong>GCKey username and password</strong>
                                    </li>
                                </ul>

                                <ToggleDetails summary={"Contact the GCKey Service Desk"}>
                                    <p>
                                        <strong>
                                            GCKey representatives can only help with GCKey issues (including two-factor
                                            authentication)
                                        </strong>
                                        . They don’t have access to any Government of Canada account or application
                                        information.
                                    </p>
                                    <ul className="list-disc ml-6">
                                        <li>Canada and the United States: 1-855-438-1102</li>
                                        <li>Text telephone (TTY/TDD): 1-855-438-1103</li>
                                        <li>Outside Canada and the United States: 1-800-2318-6290</li>
                                    </ul>
                                    <p className="mt-2">
                                        GCKey representatives are available to assist you by phone year round, 24 hours a
                                        day, 7 days a week.
                                    </p>
                                </ToggleDetails>
                            </ToggleDetails>

                            <ToggleDetails summary={"Sign-In Partner"}>
                                <p>
                                    Find help and frequently asked questions in the Sign-In Partner menu after you
                                    select <strong><i>Interac</i>® Sign-In Partner</strong>.
                                </p>
                                <p>
                                    You can also{" "}
                                    <a
                                        className="text-[#26374a] hover:text-blue-700 underline"
                                        href="https://ircc.canada.ca/english/helpcentre/results-by-topic.asp?st=23.2.1"
                                    >
                                        read our answers to common Sign-In Partner questions
                                    </a>
                                    .
                                </p>
                            </ToggleDetails>
                        </ToggleDetails>
                    </div>
                </div>
            </div>
        </section>
    );
}
