"use client";
import React, { JSX, useState } from "react";
import { FaCaretRight, FaCaretDown, FaExclamationCircle } from "react-icons/fa";

interface Section {
    id: string;
    title: string;
    content?: string | JSX.Element;
    children?: Section[];
    status?: "closed" | "default";
}

const AccordionItem: React.FC<{
    section: Section;
    isOpen: boolean;
    onToggle: (id: string) => void;
}> = ({ section, isOpen, onToggle }) => (
    <div className="border border-gray-300 bg-[#dbeef5] my-3">
        {/* Header */}
        <button
            onClick={() => onToggle(section.id)}
            className={`w-full flex items-center px-1 py-2 text-left transition-colors duration-200 ${
                isOpen ? "text-blue-700" : "text-[#26374a]"
            } hover:text-blue-700`}
        >
            {isOpen ? (
                <FaCaretDown className="mr-3 text-gray-700 text-lg" />
            ) : (
                <FaCaretRight className="mr-3 text-gray-700 text-lg" />
            )}
            <span
                className={`text-lg hover:underline font-medium ${
                    isOpen ? "text-blue-700" : "text-[#26374a]"
                } hover:text-blue-700`}
            >
                {section.title}
            </span>

            {section.status === "closed" && (
                <span className="ml-3 border-l-4 border-red-600 bg-[#fff1eb] text-black text-sm px-3 py-1">
                    Closed
                </span>
            )}
        </button>

        {/* Content */}
        {isOpen && (
            <div className="px-8 pb-5 text-base text-gray-800 leading-relaxed">
                {section.content}
                {section.children && (
                    <div className="mt-3">
                        {section.children.map((child) => (
                            <Accordion key={child.id} section={child} />
                        ))}
                    </div>
                )}
            </div>
        )}
    </div>
);

const Accordion: React.FC<{ section: Section }> = ({ section }) => {
    const [openId, setOpenId] = useState<string | null>(null);
    const toggle = (id: string) => setOpenId(openId === id ? null : id);

    return (
        <AccordionItem
            section={section}
            isOpen={openId === section.id}
            onToggle={toggle}
        />
    );
};

const NestedAccordion: React.FC = () => {
    const sections: Section[] = [
        {
            id: "1",
            title: "Study permit applicants",
            content: (
                <>
                    <p>
                        As of 8:30 am ET on January 22, 2024, most students must include a
                        provincial attestation letter (PAL) from the province or territory
                        where they plan to study with their study permit application.
                    </p>
                    <p>
                        In most cases, if you apply without a PAL, your application will be
                        returned with fees.
                    </p>
                    <a href="#" className="text-[#26374a] hover:text-blue-700 underline">
                        Learn more about the provincial attestation letter.
                    </a>
                </>
            ),
            children: [
                {
                    id: "1-1",
                    title: "New PAL/TAL exemptions for study permit applications",
                    content: (
                        <p>
                            The exemptions for including a PAL/TAL with your study permit
                            application have changed. If your application is already in
                            progress (but hasn’t been submitted), check the list of{" "}
                            <a href="#" className="text-[#26374a] hover:text-blue-700 underline">
                                PAL/TAL exemptions
                            </a>{" "}
                            to see if you need to include one.
                        </p>
                    ),
                },
            ],
        },
        {
            id: "2",
            title: "Delays with the status of your medical exam results",
            children: [
                {
                    id: "2-1",
                    title: "If we asked you for a medical exam after you applied",
                    content: (
                        <>
                            <p>
                                You may not see the status of your results updated right away.
                            </p>
                            <ul className="list-disc ml-6">
                                <li>
                                    Your results <b>usually</b> show up in your account{" "}
                                    <b>within 30 days</b> of the date you booked your exam.
                                </li>
                                <li>
                                    It may <b>take longer</b> (for example, if we need more
                                    information from you).
                                </li>
                            </ul>
                        </>
                    ),
                },
                {
                    id: "2-2",
                    title:
                        "If you got your medical exam before you applied (upfront medical exam)",
                    content: (
                        <>
                            <p>
                                You may not see the status of your results updated in your
                                account until we finish processing your immigration application.
                                We need time to:
                            </p>
                            <ul className="list-disc ml-6">
                                <li>process your application</li>
                                <li>link the results to your application</li>
                            </ul>
                            <a href="#" className="text-[#26374a] hover:text-blue-700 underline">
                                Check the application processing time.
                            </a>
                        </>
                    ),
                },
                {
                    id: "2-3",
                    title:
                        "If we ask or if a panel physician asks you for additional medical information, tests or treatment",
                    content: (
                        <>
                            <p>To avoid more delays, you must:</p>
                            <ul className="list-disc ml-6">
                                <li>go back to see the physician</li>
                                <li>
                                    complete the requirement <b>before</b> we can assess your
                                    medical exam
                                </li>
                            </ul>
                            <p>
                                Don’t complete a new medical exam unless we instruct you to do
                                so.
                            </p>
                            <p>
                                Your panel physician can explain how long it will take to
                                complete additional testing or treatment.
                            </p>
                        </>
                    ),
                },
            ],
        },
        {
            id: "3",
            title: "Student Direct Stream",
            status: "closed",
            content: (
                <p>The last day to apply was November 8, 2024, before 2 p.m. ET.</p>
            ),
        },
        {
            id: "4",
            title: "Francophone Minority Communities Student Pilot applicants",
            content: (
                <p>
                    You must use the IRCC secure account to apply with a family member who
                    is applying for a work permit, even if they apply at a different time.
                </p>
            ),
        },
    ];

    return (
<div className={"bg-[#dbeef5] "}>
    <div className="max-w-7xl mx-auto flex flex-col lg:flex-row justify-center items-center py-8  px-4 sm:px-6 lg:px-8">
        {/* Left Icon */}
        <div className="mr-8">
            <div className="bg-gray-800 text-white rounded-full p-3 flex items-center justify-center">
                <FaExclamationCircle className="text-base" />
            </div>
        </div>

        {/* Accordion Content */}
        <div className="w-full  ">
            {sections.map((section) => (
                <Accordion key={section.id} section={section} />
            ))}
        </div>
    </div>
</div>    );
};

export default NestedAccordion;
